#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
    /// </summary>
    [Description("Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  ")]
    public class Volatility : Indicator
    {
        private double aTRmultiplier = 3; // Default setting for ATRmultiplier
        private int aTRperiods = 7; // Default setting for ATRperiods
        private double vSensitivity = 1;
        private int counter = 0;
        private double SignalClose = 0;
        private DataSeries Plotset;
        private DataSeries Sideset;  //  Short == 0  Long ==1
        private bool paintbar = false;
        private bool pullbacks = true;
        private Color barColorUp = Color.Blue;
        private Color barColorDown = Color.Red;


        // Wizard generated variables           
        private double multiplier = 2.618; // Default setting for Multiplier
        private bool showArrows = true;
        // User defined variables (add any user defined variables below)
        private BoolSeries Trend;
        private DataSeries oscillator;
        private bool alert = true;
        //	private DataSeries TrendDown;
        private double _multiplier1;

        private int VTBTrend = 0;
        private int ASTTrend = 0;
        private int TrendLast = 0;

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(new Pen(Color.Green, 1), PlotStyle.Hash, "Upper"));
            Add(new Plot(new Pen(Color.Magenta, 1), PlotStyle.Hash, "Lower"));

            Plots[0].Pen.Width = 1;
            Plots[1].Pen.Width = 1;
            Plots[0].Pen.DashStyle = DashStyle.Solid;
            Plots[1].Pen.DashStyle = DashStyle.Solid;

            CalculateOnBarClose = true;
            Overlay = true;
            PriceTypeSupported = false;
            Plotset = new DataSeries(this);
            Sideset = new DataSeries(this);

            Add(new Plot(Color.Green, PlotStyle.Line, "UpTrend"));
            Add(new Plot(Color.Red, PlotStyle.Line, "DownTrend"));
        }

        protected override void OnStartUp()
        {
            Trend = new BoolSeries(this);
            oscillator = new DataSeries(this);
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
            //if (CurrentBar < (aTRperiods)) 										/// don't start until specified # of bars
            //    return;
            if (CurrentBar == (aTRperiods))										/// initiate values after 1 cycle of ATR
            {
                if (Close[0] < Close[1])										/// on downclose,
                {
                    VTBTrend = -1;
                    Sideset.Set(0);										/// start a downwave
                    counter = 1;										/// start counting bars in this wave
                    SignalClose = MIN(Close, (aTRperiods))[0];						/// set datapoint to lowest close
                    Plotset.Set(SignalClose + (vSensitivity * aTRmultiplier * (ATR(aTRperiods)[0])));	/// calculate initial upper stop value
                }
                if (Close[0] >= Close[1])									/// on upclose, 
                {
                    VTBTrend = 1;
                    Sideset.Set(1);										/// start an upwave
                    counter = 1;										/// start counting bars in this wave
                    SignalClose = MAX(Close, (aTRperiods))[0];						/// set datapoint to highest close
                    Plotset.Set(SignalClose - (vSensitivity * aTRmultiplier * (ATR(aTRperiods)[0])));	/// calculate initial lower stop value
                }
            }

            if (CurrentBar > (aTRperiods))
            {
                if (Sideset[1] == 0)											/// in downwave
                {
                    if (Close[0] < Plotset[1])								/// if no upflip
                    {
                        Sideset.Set(0);										/// continue downwave
                        SignalClose = MIN(Close, (counter))[0];					/// on lower close move datapoint down
                        Plotset.Set(SignalClose + (vSensitivity * aTRmultiplier * ATR(aTRperiods)[0]));	/// calculate stop value
                        Upper.Set(Plotset[0]);								/// set upper stop value

                    }
                    if (Close[0] >= Plotset[1])  							/// if upflip
                    {
                        VTBTrend = 1;
                        Sideset.Set(1);										/// flip to upwave
                        SignalClose = Close[0];								/// use last close as datapoint
                        counter = 2;										/// reset counter for next bar, #2 in the new wave
                        Plotset.Set(SignalClose - (vSensitivity * aTRmultiplier * ATR(aTRperiods)[0]));	/// calculate stop value
                        Lower.Set(Plotset[0]);      						/// set lower stop value
                        Upper.Set(Plotset[1]);								/// final print of upper stop
                    }
                }

                if (Sideset[1] == 1)											/// in upwave
                {
                    if (Close[0] > Plotset[1])								/// if no downflip
                    {
                        Sideset.Set(1);										/// continue upwave
                        SignalClose = MAX(Close, (counter))[0];					/// on higher close move datapoint up
                        Plotset.Set(SignalClose - (vSensitivity * aTRmultiplier * ATR(aTRperiods)[0]));	/// calculate stop value
                        Lower.Set(Plotset[0]);								/// set lower stop value
                    }

                    if (Close[0] <= Plotset[1])   								/// if downflip
                    {
                        VTBTrend = -1;
                        Sideset.Set(0);										/// flip to downwave
                        SignalClose = Close[0];								/// use last close as datapoint
                        counter = 2;										/// reset counter for next bar, #2 in the new wave
                        Plotset.Set(SignalClose + (vSensitivity * aTRmultiplier * ATR(aTRperiods)[0]));	/// calculate stop value
                        Upper.Set(Plotset[0]);    							/// set upper stop value
                        Lower.Set(Plotset[1]);								/// final print of lower stop
                    }
                }
                counter++;													/// count how many bars in each wave

                if (paintbar)
                {
                    if (Close[0] < Plotset[0] && Close[0] < Close[1])
                    {
                        BarColor = barColorDown; CandleOutlineColor = barColorDown;
                    }
                    else if (Close[0] < Plotset[0] && Close[0] > Close[1])
                    {
                        if (Pullbacks) { BarColor = Color.Transparent; CandleOutlineColor = barColorDown; }
                        else { BarColor = barColorDown; CandleOutlineColor = barColorDown; }
                    }
                    else if (Close[0] > Plotset[0] && Close[0] > Close[1])
                    {
                        BarColor = barColorUp; CandleOutlineColor = barColorUp;
                    }
                    else if (Close[0] > Plotset[0] && Close[0] < Close[1])
                    {
                        if (Pullbacks) { BarColor = Color.Transparent; CandleOutlineColor = barColorUp; }
                        else { BarColor = barColorUp; CandleOutlineColor = barColorUp; }
                    }
                }
            }
            // Use this method for calculating your indicator values. Assign a value to each
            // plot below by replacing 'Close[0]' with your own formula.
            if (CurrentBar < 20)
            {
                Trend.Set(true);
                UpTrend.Set(Close[0]);
                DownTrend.Set(Close[0]);
                return;
            }
            if (Close[0] > DownTrend[1])
            {
                Trend.Set(true);
            }
            else
            {
                if (Close[0] < UpTrend[1])
                {
                    Trend.Set(false);
                }
                else
                {
                    Trend.Set(Trend[1]);
                }
            }

            int period = (int)HomodyneDiscriminator(Median)[0];
            _multiplier1 = HomodyneDiscriminator(Median)[0] / 10;

            if (Trend[0] && !Trend[1])
            {
                UpTrend.Set(Median[0] - ATR(period)[0] * _multiplier1);
                UpTrend.Set(1, DownTrend[1]);
                ASTTrend = 1;
                TrendLast = 0;
            }
            else
            {
                if (!Trend[0] && Trend[1])
                {
                    DownTrend.Set(Median[0] + ATR(period)[0] * _multiplier1);//
                    DownTrend.Set(1, UpTrend[1]);
                    ASTTrend = -1;
                    TrendLast = 0;
                }
                else
                {
                    if (Trend[0])
                    {
                        double si = ATR(period)[0] * _multiplier1;
                        // this.HomodyneDiscriminator(Close)[0]*Multiplier;
                        UpTrend.Set((Median[0] - si) > UpTrend[1] ? (Median[0] - si) : UpTrend[1]);
                        Oscillator.Set(1.0);
                    }
                    else
                    {
                        //double si =  this.HomodyneDiscriminator(Close)[0]*Multiplier;
                        double si = ATR(period)[0] * _multiplier1;
                        DownTrend.Set((Median[0] + si) < DownTrend[1] ? (Median[0] + si) : DownTrend[1]);
                        Oscillator.Set(0.0);
                    }
                }
            }
            if (ASTTrend == 1 && VTBTrend == 1 && TrendLast != 1)
            {
                TrendLast = 1;
                //DrawText(CurrentBar.ToString(), false, "U", 0, High[0], 20, Color.Red, new Font("Arial", 12, FontStyle.Bold), StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
                if (ShowArrows)
                {
                    DrawArrowUp(CurrentBar.ToString(), true, 0, UpTrend[0] - TickSize, Color.Blue);
                }
                if (ActiveAlert)
                {
                    //Generar alarma sonora de flecha para abajo
                    Alert("AlertaVenta", NinjaTrader.Cbi.Priority.Low, "Compra autotrend", "AutoBreakEven.wav", 10, Color.Gold, Color.Green);
                }
            }
            else if (ASTTrend == -1 && VTBTrend == -1 && TrendLast != -1)
            {
                TrendLast = -1;
                //DrawText(CurrentBar.ToString(), false, "D", 0, Low[0], -20, Color.Red, new Font("Arial", 12, FontStyle.Bold), StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
                if (ShowArrows)
                {
                    DrawArrowDown(CurrentBar.ToString(), true, 0, DownTrend[0] + TickSize, Color.Red);
                }
                if (ActiveAlert)
                {
                    //Generar alarma sonora de flecha para abajo
                    Alert("AlertaVenta", NinjaTrader.Cbi.Priority.Low, "Venta autotrend", "AutoBreakEven.wav", 10, Color.Gold, Color.Red);
                }
            }


        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries UpTrend
        {
            get { return Values[2]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DownTrend
        {
            get { return Values[3]; }
        }

        [Description("Active Alert")]
        [Category("Parameters")]
        public bool ActiveAlert
        {
            get { return alert; }
            set { alert = value; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Oscillator
        {
            get { return oscillator; }
        }

        [Description("Show Arrows when Trendline is violated")]
        [Category("Parameters")]
        public bool ShowArrows
        {
            get { return showArrows; }
            set { showArrows = value; }
        }

        [Description("Period Multiplier")]
        [Category("Parameters")]
        public double Multiplier
        {
            get { return multiplier; }
            set { multiplier = Math.Max(0.0001, value); }
        }
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Upper
        {
            get { return Values[0]; }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Lower
        {
            get { return Values[1]; }
        }

        [Description("applied to ATR to calculate stop distance")]
        [Category("Parameters")]
        public double ATRmultiplier
        {
            get { return aTRmultiplier; }
            set { aTRmultiplier = Math.Max(1, value); }
        }

        [Description("Volatility Sensitivity")]
        [Category("Parameters")]
        public double VSensitivity
        {
            get { return vSensitivity; }
            set { vSensitivity = Math.Max(0, value); }
        }

        [Description("how many bars for ATR average")]
        [Category("Parameters")]
        public int ATRperiods
        {
            get { return aTRperiods; }
            set { aTRperiods = Math.Max(1, value); }
        }

        [Description("Paint Bars")]
        [Gui.Design.DisplayName("Paint Bars")]
        [Category("Visual")]
        public bool PaintBars
        {
            get { return paintbar; }
            set { paintbar = value; }
        }

        [Description("Paint Pullback bars hollow")]
        [Gui.Design.DisplayName("Show Pullbacks?")]
        [Category("Visual")]
        public bool Pullbacks
        {
            get { return pullbacks; }
            set { pullbacks = value; }
        }

        [XmlIgnore()]
        [Description("Color of Up Bar.")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Up Bar")]
        public Color BarColorUp
        {
            get { return barColorUp; }
            set { barColorUp = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string barColorUpSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorUp); }
            set { barColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }

        /// <summary>
        /// </summary>
        [XmlIgnore()]
        [Description("Color of Down Bar")]
        [Category("Visual")]
        [Gui.Design.DisplayNameAttribute("Down Bar")]
        public Color BarColorDown
        {
            get { return barColorDown; }
            set { barColorDown = value; }
        }

        /// <summary>
        /// </summary>
        [Browsable(false)]
        public string barColorDownSerialize
        {
            get { return NinjaTrader.Gui.Design.SerializableColor.ToString(barColorDown); }
            set { barColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private Volatility[] cacheVolatility = null;

        private static Volatility checkVolatility = new Volatility();

        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        public Volatility Volatility(bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            return Volatility(Input, activeAlert, aTRmultiplier, aTRperiods, multiplier, showArrows, vSensitivity);
        }

        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        public Volatility Volatility(Data.IDataSeries input, bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            if (cacheVolatility != null)
                for (int idx = 0; idx < cacheVolatility.Length; idx++)
                    if (cacheVolatility[idx].ActiveAlert == activeAlert && Math.Abs(cacheVolatility[idx].ATRmultiplier - aTRmultiplier) <= double.Epsilon && cacheVolatility[idx].ATRperiods == aTRperiods && Math.Abs(cacheVolatility[idx].Multiplier - multiplier) <= double.Epsilon && cacheVolatility[idx].ShowArrows == showArrows && Math.Abs(cacheVolatility[idx].VSensitivity - vSensitivity) <= double.Epsilon && cacheVolatility[idx].EqualsInput(input))
                        return cacheVolatility[idx];

            lock (checkVolatility)
            {
                checkVolatility.ActiveAlert = activeAlert;
                activeAlert = checkVolatility.ActiveAlert;
                checkVolatility.ATRmultiplier = aTRmultiplier;
                aTRmultiplier = checkVolatility.ATRmultiplier;
                checkVolatility.ATRperiods = aTRperiods;
                aTRperiods = checkVolatility.ATRperiods;
                checkVolatility.Multiplier = multiplier;
                multiplier = checkVolatility.Multiplier;
                checkVolatility.ShowArrows = showArrows;
                showArrows = checkVolatility.ShowArrows;
                checkVolatility.VSensitivity = vSensitivity;
                vSensitivity = checkVolatility.VSensitivity;

                if (cacheVolatility != null)
                    for (int idx = 0; idx < cacheVolatility.Length; idx++)
                        if (cacheVolatility[idx].ActiveAlert == activeAlert && Math.Abs(cacheVolatility[idx].ATRmultiplier - aTRmultiplier) <= double.Epsilon && cacheVolatility[idx].ATRperiods == aTRperiods && Math.Abs(cacheVolatility[idx].Multiplier - multiplier) <= double.Epsilon && cacheVolatility[idx].ShowArrows == showArrows && Math.Abs(cacheVolatility[idx].VSensitivity - vSensitivity) <= double.Epsilon && cacheVolatility[idx].EqualsInput(input))
                            return cacheVolatility[idx];

                Volatility indicator = new Volatility();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.ActiveAlert = activeAlert;
                indicator.ATRmultiplier = aTRmultiplier;
                indicator.ATRperiods = aTRperiods;
                indicator.Multiplier = multiplier;
                indicator.ShowArrows = showArrows;
                indicator.VSensitivity = vSensitivity;
                Indicators.Add(indicator);
                indicator.SetUp();

                Volatility[] tmp = new Volatility[cacheVolatility == null ? 1 : cacheVolatility.Length + 1];
                if (cacheVolatility != null)
                    cacheVolatility.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheVolatility = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Volatility Volatility(bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            return _indicator.Volatility(Input, activeAlert, aTRmultiplier, aTRperiods, multiplier, showArrows, vSensitivity);
        }

        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        public Indicator.Volatility Volatility(Data.IDataSeries input, bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            return _indicator.Volatility(input, activeAlert, aTRmultiplier, aTRperiods, multiplier, showArrows, vSensitivity);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Volatility Volatility(bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            return _indicator.Volatility(Input, activeAlert, aTRmultiplier, aTRperiods, multiplier, showArrows, vSensitivity);
        }

        /// <summary>
        /// Reverses when price closes beyond the stop line.  Stop line adjusts for volatility by using ATR multiplied by a constant, subtracted from the highest close (in an upwave), or added to the lowest close (in a downwave).  
        /// </summary>
        /// <returns></returns>
        public Indicator.Volatility Volatility(Data.IDataSeries input, bool activeAlert, double aTRmultiplier, int aTRperiods, double multiplier, bool showArrows, double vSensitivity)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.Volatility(input, activeAlert, aTRmultiplier, aTRperiods, multiplier, showArrows, vSensitivity);
        }
    }
}
#endregion
